(**
   A simple flashing light, to show that your application has not
   crashed (yet ;-)).
**)

MODULE VOPower;

(*
    Implements something like a flashing power LED.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       F := VOFrame,
       G := VOGUIObject,
       O := VOObject,
       P := VOPrefs;

CONST
  blinkPeriod = 100;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame* : LONGINT; (* the frame to use for the button *)
               END;


  Power*     = POINTER TO PowerDesc;
  PowerDesc* = RECORD (G.ImageDesc)
                 prefs   : Prefs;
                 frame   : F.Frame;
                 timeOut : D.TimeOut;
                 current : LONGINT;
                 up      : BOOLEAN;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
  END Init;

  PROCEDURE (p : Power) Init*;

  BEGIN
    p.Init^;

    p.prefs:=prefs;

    p.timeOut:=NIL;
    p.current:=D.blackColor;
    p.up:=TRUE;

    NEW(p.frame);
    p.frame.Init;
    p.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (p : Power) CalcSize*(display : D.Display);

  BEGIN
    p.frame.SetInternalFrame(p.prefs.frame);
    p.frame.CalcSize(display);

    p.width:=p.frame.leftBorder+2*display.spaceWidth+p.frame.rightBorder;
    p.height:=p.frame.topBorder+display.spaceHeight+p.frame.bottomBorder;

    p.minWidth:=p.width;
    p.minHeight:=p.height;

    p.CalcSize^(display);
  END CalcSize;

  PROCEDURE (p : Power) DrawBox;

  BEGIN
    p.draw.PushForeground(p.current);
    p.draw.FillRectangle(p.x+p.frame.leftBorder,p.y+p.frame.topBorder,
                         p.width-p.frame.leftBorder-p.frame.rightBorder,
                         p.height-p.frame.topBorder-p.frame.bottomBorder);
    p.draw.PopForeground;

    CASE p.current OF
      D.blackColor:
        p.up:=TRUE;
        p.current:=D.shadowColor;
    | D.shadowColor:
        IF p.up THEN
          p.current:=D.halfShadowColor;
        ELSE
          p.current:=D.blackColor;
        END;
    | D.halfShadowColor:
        IF p.up THEN
          p.current:=D.backgroundColor;
        ELSE
          p.current:=D.shadowColor;
        END;
    | D.backgroundColor:
        IF p.up THEN
          p.current:=D.halfShineColor;
        ELSE
          p.current:=D.halfShadowColor;
        END;
    | D.halfShineColor:
        IF p.up THEN
          p.current:=D.shineColor;
        ELSE
          p.current:=D.backgroundColor;
        END;
    | D.shineColor:
        IF p.up THEN
          p.current:=D.whiteColor;
        ELSE
          p.current:=D.halfShineColor;
        END;
    | D.whiteColor:
        p.current:=D.shineColor;
        p.up:=FALSE;
    END;

  END DrawBox;

  PROCEDURE (p : Power) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    p.Draw^(x,y,draw);

    p.frame.Resize(p.width,p.height);
    p.frame.Draw(p.x,p.y,draw);

    p.DrawBox;

    IF p.timeOut=NIL THEN
      p.timeOut:=p.display.AddTimeOut(0,blinkPeriod,p);
    END;
  END Draw;

  PROCEDURE (p : Power) Hide*;

  BEGIN
    IF p.timeOut#NIL THEN
      p.display.RemoveTimeOut(p.timeOut);
      p.timeOut:=NIL;
    END;

    IF p.visible THEN
      p.frame.Hide;
      p.DrawHide;
      p.Hide^;
    END;
  END Hide;

  PROCEDURE (p : Power) Receive*(msg : O.Message);

  BEGIN
    WITH msg:
      D.TimeOutMsg DO
        IF p.visible THEN
          p.DrawBox;
          p.timeOut:=p.display.AddTimeOut(0,blinkPeriod,p);
        END;
    ELSE
    END;
  END Receive;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOPower.
