(**
  A model for table-like data. This is an abstract baseclass. It just
  defines an interface which must be implemented by derived models.

  A database modl f.e. would directly work on the result set. Other
  model may work on lists as internal datarepresentation. These models may
  implement additional methods for direct access or for adding and removing
  lines or data.
**)

MODULE VOTableModel;

(*
    A model for table-like data.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT O   := VOObject,
       U   := VOUtil,

       co  := IntStr,
       str := Strings;

TYPE
  TableModel*       = POINTER TO TableModelDesc;

  (**
    Abstract baseclass for all table models. Its purpose is, to define an
    common interface for all tablemodels. The table object itself will only
    talk with the model through this interface.
  **)
  TableModelDesc*   = RECORD (O.ModelDesc)
                      END;


  (* Some internal datatstructures for the ASTableModel *)

  ASColumn          = POINTER TO ARRAY OF U.Text;
  ASRows            = POINTER TO ARRAY OF ASColumn;

  ASColumnInfoDesc  = RECORD
                        width : LONGINT;
                      END;

  ASInfo            = POINTER TO ARRAY OF ASColumnInfoDesc;
  (**

  **)

  ASTableModel*     = POINTER TO ASTableModelDesc;

  (**
    This implementation stores the value internally as a two dimentional (a)rray
    of (s) strings.
  **)
  ASTableModelDesc* = RECORD (TableModelDesc)
                        rows : ASRows;
                        info : ASInfo;
                      END;


  TestTable*        = POINTER TO TestModelDesc;

  (**
    A simple model implementation, just for testing purpose.
  **)
  TestModelDesc*    = RECORD (TableModelDesc)
                        buffer : U.Text;
                      END;

  (* --------------------------- *)

  RefreshCell*     = POINTER TO RefreshCellDesc;
  RefreshCellDesc* = RECORD (O.ResyncMsgDesc)
                       x*,y* : LONGINT;
                     END;

VAR
  refreshCell : RefreshCell; (* We allocate a global instance and reuse it everytime *)

  PROCEDURE (t : TableModel) Init*;

  BEGIN
    t.Init^;
  END Init;

  PROCEDURE (t : TableModel) GetColumns*():LONGINT;

  BEGIN
    RETURN 0;
  END GetColumns;

  PROCEDURE (t : TableModel) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    RETURN 15;
  END GetColumnWidth;

  PROCEDURE (t : TableModel) GetRows*():LONGINT;

  BEGIN
    RETURN 0;
  END GetRows;

  PROCEDURE (t : TableModel) GetText*(x,y : LONGINT):U.Text;

  BEGIN
    RETURN NIL;
  END GetText;


  PROCEDURE (t : TestTable) Init*;

  BEGIN
    t.Init^;

    NEW(t.buffer,100);
  END Init;


  PROCEDURE (t : TestTable) GetColumns*():LONGINT;

  BEGIN
    RETURN 10;
  END GetColumns;

  PROCEDURE (t : TestTable) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    RETURN 15;
  END GetColumnWidth;

  PROCEDURE (t : TestTable) GetRows*():LONGINT;

  BEGIN
    RETURN 100;
  END GetRows;

  PROCEDURE (t : TestTable) GetText*(x,y : LONGINT):U.Text;

  VAR
    help : ARRAY 32 OF CHAR;

  BEGIN
    COPY("(",t.buffer^);
    co.IntToStr(x,help);
    str.Append(help,t.buffer^);
    str.Append(",",t.buffer^);
    co.IntToStr(y,help);
    str.Append(help,t.buffer^);
    str.Append(")",t.buffer^);

    RETURN t.buffer;
  END GetText;

  (**
    Set the size of the table in rows and columns.
    Currently all previous stored texts are lost. This may change
    in the future.

    NOTE
    We make the internally used array one greater in width and height to store
    possible additional information. Currently the first row is used to store
    the column header texts.
  **)

  PROCEDURE (t : ASTableModel) SetSize*(width,height : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    NEW(t.rows,height+1);
    FOR x:=0 TO height DO
      NEW(t.rows[x],width+1);
    END;

    NEW(t.info,width+1);
    FOR x:=0 TO width DO
      t.info[x].width:=15;
    END;

    t.Notify(NIL);
  END SetSize;

  PROCEDURE (t : ASTableModel) Init*;

  BEGIN
    t.Init^;

    t.SetSize(0,0);
  END Init;

  PROCEDURE (t : ASTableModel) GetColumns*():LONGINT;

  BEGIN
    IF t.rows=NIL THEN
      RETURN 0;
    ELSE
      RETURN LEN(t.rows[0]^)-1;
    END;
  END GetColumns;

  PROCEDURE (t : ASTableModel) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    IF t.info=NIL THEN
      RETURN 15;
    ELSE
      RETURN t.info[index].width;
    END;
  END GetColumnWidth;

  PROCEDURE (t : ASTableModel) SetColumnWidth*(index,width : LONGINT);

  BEGIN
    IF t.info#NIL THEN
      t.info[index].width:=width;
    END;
  END SetColumnWidth;

  PROCEDURE (t : ASTableModel) SetColumnText*(index : LONGINT; text : U.Text);

  BEGIN
    IF t.rows#NIL THEN
      t.rows[0][index]:=text;
      t.Notify(NIL);
    END;
  END SetColumnText;

  PROCEDURE (t : ASTableModel) SetColumnString*(index : LONGINT; string : ARRAY OF CHAR);

  BEGIN
    IF t.rows#NIL THEN
      NEW(t.rows[0][index],str.Length(string)+1);
      COPY(string,t.rows[0][index]^);
      t.Notify(NIL);
    END;
  END SetColumnString;

  PROCEDURE (t : ASTableModel) GetRows*():LONGINT;

  BEGIN
    IF t.rows=NIL THEN
      RETURN 0;
    ELSE
      RETURN LEN(t.rows^)-1;
    END;
  END GetRows;

  PROCEDURE (t : ASTableModel) SetText*(x,y : LONGINT; text : U.Text);

  BEGIN
    IF y>0 THEN
      t.rows[y][x]:=text;
    END;

    (* Notify *)
    refreshCell.x:=x;
    refreshCell.y:=y;
    t.Notify(refreshCell);
  END SetText;

  PROCEDURE (t : ASTableModel) SetString*(x,y : LONGINT; string : ARRAY OF CHAR);

  BEGIN
    IF y>0 THEN
      NEW(t.rows[y][x],str.Length(string)+1);
      COPY(string,t.rows[y][x]^);
    END;

    (* Notify *)
    refreshCell.x:=x;
    refreshCell.y:=y;
    t.Notify(refreshCell);
  END SetString;

  PROCEDURE (t : ASTableModel) GetText*(x,y : LONGINT):U.Text;

  BEGIN
    RETURN t.rows[y][x];
  END GetText;

BEGIN
  NEW(refreshCell);
END VOTableModel.