/*
 * Copyright (C) 2002  Emmanuel VARAGNAT <hddtemp@guzu.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * You should have received a copy of the GNU General Public License
 * (for example COPYING); if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// Include file generated by ./configure
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Gettext includes
#if ENABLE_NLS
#include <libintl.h>
#define _(String) gettext (String)
#else
#define _(String) (String)
#endif

// Standard includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <scsi/scsi.h>
#include <scsi/sg.h>

// Application specific includes
#include "scsicmds.h"

static void scsi_fixstring(unsigned char *s, int bytecount)
{
  unsigned char *p;
  unsigned char *end;

  p = s;
  end = s + bytecount;

  /* strip leading blanks */
  while (s != end && *s == ' ')
    ++s;
  /* compress internal blanks and strip trailing blanks */
  while (s != end && *s) {
    if (*s++ != ' ' || (s != end && *s && *s != ' '))
      *p++ = *(s-1);
  }
  /* wipe out trailing garbage */
  while (p != end)
    *p++ = '\0';
}

int scsi_SG_IO(int device, unsigned char *cdb, int cdb_len, unsigned char *buffer, int buffer_len, unsigned char *sense, unsigned char sense_len, int dxfer_direction) {
  struct sg_io_hdr io_hdr;

  memset(&io_hdr, 0, sizeof(struct sg_io_hdr));
  io_hdr.interface_id = 'S';
  io_hdr.cmdp = cdb;
  io_hdr.cmd_len = cdb_len;
  io_hdr.dxfer_len = buffer_len;
  io_hdr.dxferp = buffer;
  io_hdr.mx_sb_len = sense_len;
  io_hdr.sbp = sense;
  io_hdr.dxfer_direction = dxfer_direction;
  io_hdr.timeout = 3000; /* 3 seconds should be ample */

  return ioctl(device, SG_IO, &io_hdr);
}

int scsi_inquiry(int device, unsigned char *buffer)
{
  unsigned char cdb[6];
 
  memset(cdb, 0, sizeof(cdb));
  cdb[0] = INQUIRY;
  cdb[4] = 36;  /* should be 36 for unsafe devices (like USB mass storage stuff)
                 *      otherwise they can lock up! SPC sections 7.4 and 8.6 */

  if (scsi_SG_IO(device, cdb, sizeof(cdb), buffer, cdb[4], NULL, 0, SG_DXFER_FROM_DEV) != 0)
    return 1;
  else {
    scsi_fixstring(buffer + 8, 24);
    return 0;
  }
}

int scsi_modesense(int device, unsigned char pagenum, unsigned char *buffer, int buffer_len) {
  unsigned char cdb[6];
  
  memset(cdb, 0, sizeof(cdb));
  cdb[0] = MODE_SENSE;
  cdb[2] = pagenum;
  cdb[4] = 0xff;

  return scsi_SG_IO(device, cdb, sizeof(cdb), buffer, buffer_len, NULL, 0, SG_DXFER_FROM_DEV);
}

int scsi_modeselect(int device, char *buffer) {
  unsigned char cdb[6];
  
  memset(cdb, 0, sizeof(cdb));
  cdb[0] = MODE_SELECT;
  cdb[1] = 0x11;
  cdb[4] = buffer[0] + 1;
  
  buffer[3]  = 0x08;
  buffer[10] = 0x02;
  buffer[20] &= 0x3f;
	  
  return scsi_SG_IO(device, cdb, sizeof(cdb), buffer, cdb[4], NULL, 0, SG_DXFER_TO_DEV);
}

int scsi_logsense(int device, int pagenum, unsigned char *buffer, int buffer_len) {
  unsigned char cdb[10];
  
  memset(cdb, 0, sizeof(cdb));
  cdb[0] = LOG_SENSE;
  cdb[2] = 0x40 | pagenum;
  cdb[7] = 0x04;
  
  return scsi_SG_IO(device, cdb, sizeof(cdb), buffer, buffer_len, NULL, 0, SG_DXFER_FROM_DEV);
}

int scsi_smartsupport(int device) {
  unsigned char buf[255];
	
  if (scsi_modesense (device, EXCEPTIONS_CONTROL_PAGE, buf, sizeof(buf)) != 0)
    return 0;
  else
    return (buf[14] & 0x08) == 0;
}

int scsi_smartDEXCPTdisable(int device) {
  unsigned char buf[255];

  if (scsi_modesense (device, EXCEPTIONS_CONTROL_PAGE, buf, sizeof(buf)) != 0)
    return 1;

  buf[14] &= 0xf7;
  buf[15] = 0x04;

  return scsi_modeselect (device, buf);
}

