/*
 * Copyright © 2007 Keith Packard
 * Modified by Joey Hess for hotkeys.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */
#include <stdio.h>
#include <stdlib.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/extensions/Xrandr.h>

#include "hotkeys.h"

static Atom backlight;

static long
backlight_get (Display *dpy, RROutput output)
{
    unsigned long   nitems;
    unsigned long   bytes_after;
    unsigned char   *prop;
    Atom	    actual_type;
    int		    actual_format;
    long	    value;
    
    if (XRRGetOutputProperty (dpy, output, backlight,
			      0, 4, False, False, None,
			      &actual_type, &actual_format,
			      &nitems, &bytes_after, &prop) != Success)
	return -1;
    if (actual_type != XA_INTEGER || nitems != 1 || actual_format != 32)
	value = -1;
    else
	value = *((long *) prop);
    XFree (prop);
    return value;
}

static void
backlight_set (Display *dpy, RROutput output, long value)
{
    XRRChangeOutputProperty (dpy, output, backlight, XA_INTEGER, 32,
			     PropModeReplace, (unsigned char *) &value, 1);
}

int
xbacklight (Display *dpy, int change)
{
    int     initialized = 0;
    int	    screen;
    int	    major, minor;
    int	    value = 0;

    if (! initialized) {
	    if (!XRRQueryVersion (dpy, &major, &minor))
	    {
		uError("RandR extension missing");
		return -1;
	    }
	    if (major < 1 || (major == 1 && minor < 2))
	    {
		uError("RandR version %d.%d too old", major, minor);
		return -1;
	    }
	    backlight = XInternAtom (dpy, "BACKLIGHT", True);
	    if (backlight == None)
	    {
		uError("No outputs have backlight property");
		return -1;
	    }
	    initialized = 1;
    }

    for (screen = 0; screen < ScreenCount (dpy); screen++)
    {
	Window		    root = RootWindow (dpy, screen);
	XRRScreenResources  *resources = XRRGetScreenResources (dpy, root);
	int		    o;
	
	if (!resources) continue;
	for (o = 0; o < resources->noutput; o++)
	{
	    RROutput	output = resources->outputs[o];
	    XRRPropertyInfo *info;
	    double    	cur, new;
	    double	min, max;
	    double	set;

	    cur = backlight_get (dpy, output);
	    if (cur != -1)
	    {
		info = XRRQueryOutputProperty (dpy, output, backlight);
		if (info)
		{
		    if (info->range && info->num_values == 2)
		    {
			min = info->values[0];
			max = info->values[1];
			
			set = value * (max - min) / 100;
			new = cur + change;
			
			if (new > max) new = max;
			if (new < min) new = min;
			backlight_set (dpy, output, (long) new);
		    }
		    XFree (info);
		}
	    }
	}

	XRRFreeScreenResources (resources);
    }

    return 0;
}
