///////
   //    MySQL Database class for ht://Check
   //    File: HtmysqlDB.cc
   //
   //    Copyright (c) 1999-2002 Comune di Prato - Prato - Italy
   //    Author: Gabriele Bartolini - Prato - Italy <angusgb@users.sourceforge.net>
   //
   //    For copyright details, see the file COPYING in your distribution
   //    or the GNU General Public License version 2 or later 
   //    <http://www.gnu.org/copyleft/gpl.html>
   //
   //    $Id: HtmysqlDB.cc,v 1.46 2002/02/18 10:44:59 angusgb Exp $
   //
   //    Started: 28.06.1999
///////

#include "HtmysqlDB.h"
#include <stdlib.h>
#include <ctype.h>

// Local function (declared as static)
static bool getmysqlconfvalue(const char *source, const char *pattern,
   String &destination);

#define URL_INDEX_LENGTH 64

///////
   //    Construction
///////

#if 0   
HtmysqlDB::HtmysqlDB (const String &host, const String &db,
         const String &user, const String &passwd)
{

   MySQLHost = host;
   MySQLDB = db;
   MySQLUser = user;
   MySQLPasswd = passwd;

   // Set the default length for the Index regarding
   // the Url field in the Schedule and Url tables
   URL_Index_Length = URL_INDEX_LENGTH;
   
}
#endif

HtmysqlDB::HtmysqlDB(const String &db, const String &File, String Group,
   int *argc, char ***argv)
{
   MySQLHost = 0;
   MySQLUser = 0;
   MySQLPasswd = 0;
   MySQLDB = db;
   MySQLPort = 0;
   MySQLSocket = 0;

   // Set the default length for the Index regarding
   // the Url field in the Schedule and Url tables
   URL_Index_Length = URL_INDEX_LENGTH;
   
   LoadDefaults(File, Group, argc, argv);
   
   DBSignature = MySQLDB;
   if (MySQLHost.length())
      DBSignature << "@" << MySQLHost;
   if (MySQLPort)
      DBSignature << ":" << MySQLPort;
   
}


///////
   //    Destruction
///////

HtmysqlDB::~HtmysqlDB ()
{
}

void HtmysqlDB::LoadDefaults(const String &File,
   MYSQL_LOAD_DEFAULTS_ARGTWO Group, int *argc, char ***argv)
{

   MYSQL_LOAD_DEFAULTS_ARGTWO group[] = {0, 0};
   String strPort = 0;
   
   group[0] = Group;
   
   // Use the MySQL function for getting the default settings
   // for the connection
   
   if (debug >0)
      cout << " Reading default MySQL option file: " << endl;

   load_defaults((const char*)File, group, argc, argv);

   for (int i=0; i < *argc; i++)
   {
      if (getmysqlconfvalue((*argv)[i], "user", MySQLUser))
      {
         if (debug > 0)
            cout << "  Found MySQL User: " << MySQLUser << endl;
      }
      else if (getmysqlconfvalue((*argv)[i], "host", MySQLHost))
      {
         if (debug > 0)
            cout << "  Found MySQL Host: " << MySQLHost << endl;
      }
      else if (getmysqlconfvalue((*argv)[i], "password", MySQLPasswd))
      {
         if (debug > 0)
            cout << "  Found MySQL Password: <shhh>" << endl;
      }
      else if (getmysqlconfvalue((*argv)[i], "port", strPort))
      {
         MySQLPort = atoi((char *)strPort);
         
         if (debug > 0)
            cout << "  Found MySQL Port Number: " << MySQLPort << endl;
      }
      else if (getmysqlconfvalue((*argv)[i], "socket", MySQLSocket))
      {
         if (debug > 0)
            cout << "  Found MySQL Socket: " << MySQLSocket << endl;
      }
   }

}


bool getmysqlconfvalue(const char * source, const char *pattern,
   String &destination)
{
   const char *p=source;

   for (; *p && *p=='-'; p++);   // Skip the trailing '-'

   if (!mystrncasecmp(pattern, p, strlen(pattern)))
   {
      destination.trunc();

         p+=strlen(pattern)+1;   // Go over the '='

      for (; *p && isspace(*p); p++);
      
      destination = p;
      
      return true;
   }
   
   return false;
}


///////
   //    Connection to the host specified with user and password.
   //    Here we don't connect to a precise database ... We will do it
   //    later using SelectDB method.
///////

int HtmysqlDB::Connect ()
{

   char *host;
   char *user;
   char *passwd;
   char *socket;

///////
   //    Converting empty strings into NULL pointer
   //    This allows MySQL server to act as default
///////

   if (! MySQLHost.length()) host=NULL;
   else host = MySQLHost;

   if (! MySQLUser.length()) user=NULL;
   else user = MySQLUser;

   if (! MySQLPasswd.length()) passwd=NULL;
   else passwd = MySQLPasswd;

   if (! MySQLSocket.length()) socket=NULL;
   else socket = MySQLSocket;

   if (debug > 0)
      cout << "Connecting to MySQL server on " << (host?host:"localhost")
         << " as " << (user?user:"session") << " user" << endl;

///////
   //    Connecting using Htmysql::Connect Interface method
///////

   if (! Htmysql::Connect (host, user, passwd, 0, MySQLPort, socket))
      return 0;   // Something has gone wrong ... Let's check it outside here

   return 1;   
}


///////
   //    Creating a new database for ht://Check
   //    Return 0 if an error occured, 1 OK
///////

int HtmysqlDB::CreateDatabase()
{

   String SQLStatement = 0;
   int result;

   // This variable holds the length of the URL Index
   // in the URL and Schedule tables   
   String SQL_Url_Index = "INDEX Idx_Url (Url";
   
   if (URL_Index_Length > 0)
   {
      // Let's set the length
      SQL_Url_Index << "(" << URL_Index_Length << ")";
   }

   SQL_Url_Index << ")";
   
   
///////
   //    Check if the db already exists
///////

   result = Exists (MySQLDB);
   
   if (result == -1) return 0;   // An error occured
   
   // Database already exists
   // Let's drop it before creating a new one with the same name
   if (result > 0) DropDatabase();


///////
   //    Database creation
///////
   
   if (debug >0)
      cout << "Creating ht://Check database '" << MySQLDB << "'" << endl;
   
   SQLStatement << "CREATE DATABASE " << MySQLDB;
   
   // Executing Database creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   
   // Select the database
   SelectDB (MySQLDB);

   if (debug >0)
      cout << "Database '" << MySQLDB << "' now selected" << endl;
   
///////
   //    Creation of the 'Schedule' table
///////

   if (debug >1)
      cout << " |- Creating 'htCheck' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".htCheck (" << "\n"
      << "  StartTime DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL, " << "\n"
      << "  EndTime DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL, " << "\n"
      << "  ScheduledUrls MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  TotUrls MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  RetrievedUrls MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  TCPConnections MEDIUMINT UNSIGNED  DEFAULT '0' NOT NULL, " << "\n"
      << "  ServerChanges MEDIUMINT UNSIGNED  DEFAULT '0' NOT NULL, " << "\n"
      << "  HTTPRequests MEDIUMINT UNSIGNED  DEFAULT '0' NOT NULL, " << "\n"
      << "  HTTPSeconds MEDIUMINT UNSIGNED  DEFAULT '0' NOT NULL, " << "\n"
      << "  HTTPBytes BIGINT UNSIGNED  DEFAULT '0' NOT NULL, " << "\n"
      << "  User VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  PRIMARY KEY (StartTime, EndTime)" << "\n"
      << ")" << "\n";

   // Executing Table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'Schedule' table
///////

   if (debug >1)
      cout << " |- Creating 'Schedule' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".Schedule (" << "\n"
      << "  IDUrl MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  IDServer SMALLINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  Url VARCHAR(255) BINARY DEFAULT '' NOT NULL, " << "\n"
      << "  Status ENUM('ToBeRetrieved', 'Retrieved', 'CheckIfExists',"
      << " 'Checked', 'BadQueryString', 'BadExtension', 'MaxHopCount',"
      << " 'FileProtocol', 'EMail', 'Javascript', 'NotValidService',"
      << " 'Malformed') "
             << "DEFAULT 'ToBeRetrieved' NOT NULL, " << "\n"
      << "  CreationTime DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL, " << "\n"
      << "  IDReferer MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  HopCount TINYINT UNSIGNED DEFAULT '0' NOT NULL, " << "\n"
      << "  PRIMARY KEY (IDUrl), " << "\n"
      << "  INDEX Idx_IDServer (IDServer), " << "\n"
      << "  " << SQL_Url_Index << ", " << "\n"
      << "  INDEX Idx_Status (Status) " << "\n"
      << ")" << "\n";

   // Executing Table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'Server' table
///////

   if (debug >1)
      cout << " |- Creating 'Server' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".Server (" << "\n"
      << "  IDServer SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Server VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  IPAddress VARCHAR(15)," << "\n"
      << "  Port SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  HttpServer VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  HttpVersion VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  PersistentConnection TINYINT(1) UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Requests SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  PRIMARY KEY (IDServer), " << "\n"
      << "  INDEX Idx_Server (Server(24)), " << "\n"
      << "  INDEX Idx_Requests (Requests) " << "\n"
      << ")" << "\n";

   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'Url' table
///////

   if (debug >1)
      cout << " |- Creating 'Url' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".Url (" << "\n"
      << "  IDUrl MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  IDServer SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Url VARCHAR(255) BINARY DEFAULT '' NOT NULL," << "\n"
      << "  ContentType VARCHAR(32) DEFAULT '' NOT NULL," << "\n"
      << "  ConnStatus ENUM('OK', 'NoHeader', 'NoHost', 'NoPort', "
         << "'NoConnection', 'ConnectionDown', 'ServiceNotValid', "
         << "'OtherError') "
         << "DEFAULT 'OK' NOT NULL, " << "\n"
      << "  ContentLanguage VARCHAR(16) DEFAULT '' NOT NULL," << "\n"
      << "  TransferEncoding VARCHAR(32) DEFAULT '' NOT NULL," << "\n"
      << "  LastModified DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL," << "\n"
      << "  LastAccess DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL," << "\n"
      << "  Size INT DEFAULT '0' NOT NULL," << "\n"
      << "  StatusCode SMALLINT DEFAULT '0' NOT NULL," << "\n"
      << "  ReasonPhrase VARCHAR(32) DEFAULT '' NOT NULL," << "\n"
      << "  Location VARCHAR(255) BINARY DEFAULT '' NOT NULL," << "\n"
      << "  Title VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  SizeAdd INT DEFAULT '0' NOT NULL," << "\n"
      << "  PRIMARY KEY (IDUrl), " << "\n"
      << "  INDEX Idx_IDServer (IDServer), " << "\n"
      << "  " << SQL_Url_Index << ", " << "\n"
      << "  INDEX Idx_ContentType (ContentType(16)), " << "\n"
      << "  INDEX Idx_StatusCode (StatusCode) " << "\n"
      << ")" ;

   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'HtmlStatement' table
///////

   if (debug >1)
      cout << " |- Creating 'HtmlStatement' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".HtmlStatement (" << "\n"
      << "  IDUrl MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  TagPosition SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Tag VARCHAR(32) DEFAULT '' NOT NULL," << "\n"
      << "  Statement VARCHAR(255)," << "\n"
      << "  PRIMARY KEY (IDUrl, TagPosition)," << "\n"
      << "  INDEX Idx_Tag (Tag(4))" << "\n"
      << ")" << "\n";

   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'HtmlAttribute' table
///////

   if (debug >1)
      cout << " |- Creating 'HtmlAttribute' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".HtmlAttribute (" << "\n"
      << "  IDUrl  MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  TagPosition SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  AttrPosition TINYINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Attribute VARCHAR(32) DEFAULT '' NOT NULL," << "\n"
      << "  Content VARCHAR(255)," << "\n"
      << "  PRIMARY KEY (IDUrl, TagPosition, AttrPosition)," << "\n"
      << "  INDEX Idx_Attribute (Attribute(8))" << "\n"
      << ")" << "\n";

   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured


///////
   //    Creation of the 'Link' table
///////

   if (debug >1)
      cout << " |- Creating 'Link' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".Link (" << "\n"
      << "  IDUrlSrc MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  IDUrlDest MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  TagPosition SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  AttrPosition TINYINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Anchor VARCHAR(255) BINARY DEFAULT '' NOT NULL," << "\n"
      << "  LinkType ENUM('Normal', 'Direct', 'Redirection') "
         << "DEFAULT 'Normal' NOT NULL, " << "\n"
      << "  LinkResult "
         << "ENUM('NotChecked', 'NotRetrieved', 'OK', 'Broken', "
         << "'AnchorNotFound', 'Redirected', 'NotAuthorized', "
	 << "'EMail', 'Javascript') "
         << "DEFAULT 'NotChecked' NOT NULL, " << "\n"
      << "  PRIMARY KEY (IDUrlSrc, IDUrlDest, TagPosition, AttrPosition)" << "\n"
      << ")" << "\n";


   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

///////
   //    Creation of the 'Cookies' table
///////

   if (debug >1)
      cout << " |- Creating 'Cookies' table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".Cookies (" << "\n"
      << "  IDCookie MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Name VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  Value TEXT DEFAULT '' NOT NULL," << "\n"
      << "  Path VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  Domain VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  SrcUrl VARCHAR(255) DEFAULT '' NOT NULL," << "\n"
      << "  Expires DATETIME DEFAULT '0000-00-00 00:00:00' NOT NULL, " << "\n"
      << "  Secure TINYINT DEFAULT '0' NOT NULL, " << "\n"
      << "  DomainValid TINYINT DEFAULT '0' NOT NULL, " << "\n"
      << "  PRIMARY KEY (IDCookie)" << "\n"
      << ")" << "\n";


   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

///////
   //    Creation of the 'TmpAnchors' temporary table
///////

   if (debug >1)
      cout << " |- Creating 'TmpAnchors' temporary table" << endl;


   // Write the SQL statement
   
   SQLStatement.trunc(); // Initialize it again.
   SQLStatement << "CREATE TABLE " << MySQLDB << ".TmpAnchors (" << "\n"
      << "  IDUrl MEDIUMINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  TagPosition SMALLINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  AttrPosition TINYINT UNSIGNED DEFAULT '0' NOT NULL," << "\n"
      << "  Anchor VARCHAR(255) BINARY DEFAULT '' NOT NULL," << "\n"
      << "  PRIMARY KEY (IDUrl, TagPosition, AttrPosition), " << "\n"
      << "  INDEX Idx_Anchor (Anchor(16))" << "\n"
      << ")" << "\n";

   // Executing table creation 

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;

}


///////
   //    Dropping a database for ht://Check
   //    Return 0 if an error occured, 1 OK
///////

int HtmysqlDB::DropDatabase()
{

   String SQLStatement="";
   
   if (debug >0)
      cout << "Dropping a previous ht://Check database called '"
      << MySQLDB << "'" << endl;

   SQLStatement << "DROP DATABASE " << MySQLDB;

   // Executing Database dropping

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Set the SQL BIG TABLES option (for huge queries)
///////

int HtmysqlDB::SetSQLBigTableOption ()
{

   if (debug >0)
      cout << "Setting option for big tables" << endl;
   
   String SQLStatement= "SET OPTION SQL_BIG_TABLES = 1";
   if (Query (SQLStatement) == -1)
      return 0;

   return 1;   
}


///////
   //    Optimize all the tables of the Database
///////

int HtmysqlDB::Optimize()
{

   String SQLStatement=0;
   HtDateTime OptimizeTime;
   
   if (debug >0)
      cout << "Optimizing Database '" << MySQLDB << "'"
         << " - " << OptimizeTime.GetAscTime()<< endl;
   
///////
   //    Optimization of the 'Schedule' table
///////

   if (debug >1)
      cout << " |- 'Schedule' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;

   SQLStatement = "OPTIMIZE TABLE Schedule";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'Url' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'Url' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;

   SQLStatement = "OPTIMIZE TABLE Url";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'HtmlStatement' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'HtmlStatement' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE HtmlStatement";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'HtmlAttribute' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'HtmlAttribute' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE HtmlAttribute";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'Link' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'Link' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE Link";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'Server' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'Server' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE Server";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'Cookies' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'Cookies' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE Cookies";
   if (Query (SQLStatement) == -1)
      return 0;

///////
   //    Optimization of the 'htCheck' table
///////

   OptimizeTime.SettoNow();
   if (debug >1)
      cout << " |- 'htCheck' table optimization"
         << " - " << OptimizeTime.GetAscTime()<< endl;
      
   SQLStatement = "OPTIMIZE TABLE htCheck";
   if (Query (SQLStatement) == -1)
      return 0;

   return 1;
   
}

///////
   //    Insert a Server into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const _Server &server)
{

   String SQLStatement="";
   
   if (debug >4)
      cout << "Inserting a new server into '"
      << MySQLDB << "': " << server.host() << ":" << server.port() << endl;

   SQLStatement << "Insert into " << MySQLDB << ".Server ( "
      << "IDServer, Server, IPAddress, Port, HttpServer, "
      << "HttpVersion, PersistentConnection, Requests"
      << " ) values ( "
      << server.GetID() << ", ";

   // Escape safe host name
   AppendSQLTextField(SQLStatement, server.host());

   SQLStatement << ", "
      << "'" << server.GetIPAddress() << "'" << ", "
      << server.port() << ", ";

   // Escape safe host HTTP server
   AppendSQLTextField(SQLStatement, server.GetHttpServer());

   SQLStatement << ", ";

   // Escape safe host HTTP version
   AppendSQLTextField(SQLStatement, server.GetHttpVersion());

   SQLStatement << ", "
      << server.IsPersistentConnectionAllowed() << ", "
      << server.GetRequests()
      << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Insert an Url into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const _Url &url)
{

   String SQLStatement="";
   String Status="";
   
   // Retrieve the String value of the Status
   url.RetrieveConnStatus(Status);   
   
   if (debug >4)
      cout << "Inserting a new url into '"
      << MySQLDB << "': " << url.get() << endl;

   SQLStatement << "Insert into " << MySQLDB << ".Url ( "
      << "IDUrl, IDServer, Url, ContentType, ConnStatus, TransferEncoding, "
      << "LastModified, LastAccess, Size, StatusCode, "
      << "ReasonPhrase, Location, Title, ContentLanguage"
      << " ) values ( "
      << url.GetID() << ", "
      << url.GetIDServer() << ", ";

   // Escape safe URL
   AppendSQLTextField(SQLStatement, url.get());

   SQLStatement << ", ";

   // Escape safe URL content-type
   AppendSQLTextField(SQLStatement, url.GetContentType());

   SQLStatement << ", "
      << "'" << Status << "'" << ", ";

   // TransferEncoding
   AppendSQLTextField(SQLStatement, url.GetTransferEncoding());

   SQLStatement << ", "
      << "'" << (url.GetLastModified()?(url.GetLastModified())->GetTimeStamp():"") << "'" << ", "
      << "'" << (url.GetLastAccess()?(url.GetLastAccess())->GetTimeStamp():"") << "'" << ", "
      << url.GetSize() << ", "
      << url.GetStatusCode() << ", ";

   // Reason Phrase
   AppendSQLTextField(SQLStatement, url.GetReasonPhrase());

   SQLStatement << ", ";

   // Url redirection
   AppendSQLTextField(SQLStatement, url.GetLocation());

   SQLStatement << ", ";

   // Url Title
   AppendSQLTextField(SQLStatement, url.GetTitle());

   SQLStatement << ", ";

   // ContentLanguage
   AppendSQLTextField(SQLStatement, url.GetContentLanguage());

   SQLStatement << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Insert a Schedule into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(SchedulerEntry &s)
{

   String SQLStatement="";
   String Status="";

   // Retrieve the String value of the Status
   s.RetrieveStatus(Status);   
   
   if (debug >4)
      cout << "Inserting a new scheduler entry into '"
      << MySQLDB << "': " << s << endl;

   SQLStatement << "Insert into " << MySQLDB << ".Schedule ( "
      << "IDUrl, IDServer, Url, CreationTime, Status, IDReferer, HopCount"
      << " ) values ( "
      << s.GetIDSchedule() << ", "
      << s.GetIDServer() << ", ";

   AppendSQLTextField(SQLStatement, s.GetScheduleUrl());

   SQLStatement << ", "
      << "NOW(), "
      << "'" << Status << "', "
      << s.GetIDReferer() << ", "
      << s.GetHopCount() << " "
      << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   // Retrieve the ID (cos it's auto incremental)
   s.SetIDSchedule(GetLastID());
   
   return 1;
}


///////
   //    Insert the info into the htCheck table
///////

int HtmysqlDB::Insert(const RunInfo &runinfo)
{

   String SQLStatement="";

   if (debug >4)
      cout << "Inserting the retrieval info into '"
      << MySQLDB << "'" << endl;

   SQLStatement << "Insert into " << MySQLDB << ".htCheck ( "
      << "StartTime, EndTime, ScheduledUrls, TotUrls, "
      << "RetrievedUrls, TCPConnections, ServerChanges, "
      << "HTTPRequests, HTTPSeconds, HTTPBytes, User"
      << " ) values ( "
      << "'" << runinfo.StartTime.GetTimeStamp() << "', "
      << "'" << runinfo.FinishTime.GetTimeStamp() << "', "
      << runinfo.ScheduledUrls << ", "
      << runinfo.TotUrls << ", "
      << runinfo.RetrievedUrls << ", "
      << runinfo.TCPConnections << ", "
      << runinfo.ServerChanges << ", "
      << runinfo.HTTPRequests << ", "
      << runinfo.HTTPSeconds << ", "
      << runinfo.HTTPBytes << ", "
      << "USER()"
      << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
   
}


///////
   //    Insert a Cookie into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const HtCookie &cookie)
{
   static unsigned id_cookie = 0;
   String SQLStatement="";

   ++id_cookie;   // Increment the index
      
   if (debug >4)
      cout << "Inserting a new cookie into '"
      << MySQLDB << "': " << cookie.GetName() << endl;

   SQLStatement << "Insert into " << MySQLDB << ".Cookies ( "
      << "IDCookie, Name, Value, Path, Domain, SrcUrl, Secure, DomainValid";
   
   if (cookie.GetExpires())
      SQLStatement << ", Expires";
      
   SQLStatement << ") values ( "
      << id_cookie << ", ";

   // Escape safe cookie name
   AppendSQLTextField(SQLStatement, cookie.GetName());

   SQLStatement << ", ";

   // Escape safe cookie value
   AppendSQLTextField(SQLStatement, cookie.GetValue());

   SQLStatement << ", ";

   // Escape safe cookie path
   AppendSQLTextField(SQLStatement, cookie.GetPath());

   SQLStatement << ", ";

   // Escape safe cookie path
   AppendSQLTextField(SQLStatement, cookie.GetDomain());

   SQLStatement << ", ";

   // Escape safe cookie source URL
   AppendSQLTextField(SQLStatement, cookie.GetSrcURL());

   SQLStatement << ", "
      << (cookie.getIsSecure()?1:0);

   SQLStatement << ", "
      << (cookie.getIsDomainValid()?1:0);

   if (cookie.GetExpires())
      SQLStatement << ", '" << cookie.GetExpires()->GetTimeStamp() << "'";

      
   SQLStatement << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Look for a schedule entries, given a filter and a
   //     HtmysqlQuery Result object. Returns -1 if an error occurs
   //     else returns the number of records found.
///////

int HtmysqlDB::Search(SchedulerEntry &filter, HtmysqlQueryResult &result)
{

   String SQLStatement="";

   // SQL statement construction
   
   SQLStatement << "Select "
      << "IDUrl"     << ", "     // Url identifier
      << "IDServer"  << ", "     // Server identifier
      << "Url"       << ", "     // Url
      << "Status"    << ", "     // Schedule Status
      << "IDReferer" << ", "     // ID of the referring Url
      << "HopCount"  << "  "     // Hop Count
      << " from Schedule ";

   // Create the SQL 'Where' statement given a filter
   CreateFilter(SQLStatement, filter);

   // Execute and store the Query
   
   return Query(SQLStatement, &result);

}


///////
   //    Get next Element from the query result
   //    Returns 0 if the end has been reached.
///////

int HtmysqlDB::GetNextElement (SchedulerEntry &dest, HtmysqlQueryResult &result)
{

   MYSQL_ROW Rows;
   
   if ((Rows = result.GetNextRecord()) == 0) // end of query reached
      return 0;

   // Reset the destination
   dest.Reset();

   // Set the ID      
   dest.SetIDSchedule(atoi(Rows[0]));

   // Set the ID of the server
   dest.SetIDServer(atoi(Rows[1]));
   
   // Set the Url
   dest.SetScheduleUrl((char *)Rows[2]);

   // Set the Status
   dest.SetStatus((char *)Rows[3]);
      
   // Set the ID of the Referring URL
   dest.SetIDReferer(atoi(Rows[4]));
      
   // Set the hop count number
   dest.SetHopCount(atoi(Rows[5]));
      
   return 1;
}


///////
   //    Create a SQL filter string on a Scheduler Entry
///////

void HtmysqlDB::CreateFilter(String &SQLStatement, SchedulerEntry &filter)
{
   // Search for any filter to be applied to the query
   
   int flag = 0;

   // Applying Url Identifier
      
   if (filter.GetIDSchedule() != 0)   // specified an ID Url
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "IDUrl = " << filter.GetIDSchedule() << " ";
   }


   // Applying Server
      
   if (filter.GetIDServer() != 0)   // specified a server
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "IDServer = " << filter.GetIDServer() << " ";
   }
         

   // Applying Url
      
   if (filter.GetScheduleUrl().length())   // specified an Url
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "Url = ";
   
      AppendSQLTextField(SQLStatement, filter.GetScheduleUrl());
      
   }
         

   // Applying Status
      

   if (filter.GetStatus() != SchedulerEntry::Url_Empty)
   {
      // A Status has been specified
      // Retrieve the String value of the Status
      String Status="";

      filter.RetrieveStatus(Status);   
      
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "Status = '" << Status << "' ";
   }

   // Applying Referring Url ID
      
   if (filter.GetIDReferer() != 0)   // specified an ID Referer Url
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "IDReferer = " << filter.GetIDReferer() << " ";
   }

}


///////
   //    Execute a generic query and stores the results
   //    Returns -1 if an error occured
   //    Else the number of rows retrieved if we want to store a result
   //    or 0 if not.
///////

int HtmysqlDB::Query(String &SQLStatement, HtmysqlQueryResult *result,
   Query_Type qt)
{
   
   if (debug >5)
      cout << "SQL Statement: " << SQLStatement << endl;

   // Executing Select query

   if ( ExecQuery ( (char *) SQLStatement))
      return -1;   // An error occured

   // Stores the result of the query
   if (result)
   {
      switch(qt)
      {
         case Htmysql_Stored: // stored query - slower
               if (debug>5) cout << "Stored query" << endl;
               if (! StoreResult(*result)) return -1;
               else return result->GetRows();
               break;
         case Htmysql_Temporary: // faster
               if (debug>5) cout << "Direct query" << endl;
               if (! UseResult(*result)) return -1;
               else return 0;
               break;
      }
      
   }

   return 0;

}


///////
   //    Insert a HtmlStatement into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const HtmlStatement& htmlstatement)
{

   String SQLStatement="";
   
   if (debug >4)
      cout << "Inserting a new HtmlStatement into '"
      << MySQLDB << "': " << htmlstatement << endl;

   SQLStatement << "Insert into " << MySQLDB << ".HtmlStatement ( "
      << "IDUrl, TagPosition, Tag, Statement"
      << " ) values ( "
      << htmlstatement.GetIDUrl() << ", "
      << htmlstatement.GetTagPosition() << ", ";

   AppendSQLTextField(SQLStatement, htmlstatement.GetTag());
   
   SQLStatement << ", ";

   AppendSQLTextField(SQLStatement, htmlstatement.GetStatement());

   SQLStatement << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Insert a HtmlAttribute into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const HtmlAttribute& htmlattribute)
{

   String SQLStatement="";
   
   if (debug >4)
      cout << "Inserting a new HtmlAttribute into '"
      << MySQLDB << "': " << htmlattribute << endl;

   SQLStatement << "Insert into " << MySQLDB << ".HtmlAttribute ( "
      << "IDUrl, TagPosition, AttrPosition, Attribute, Content"
      << " ) values ( "
      << htmlattribute.GetIDUrl() << ", "
      << htmlattribute.GetTagPosition() << ", "
      << htmlattribute.GetAttrPosition() << ", ";
      
   AppendSQLTextField(SQLStatement, htmlattribute.GetAttribute());

   SQLStatement << ", ";
   
   AppendSQLTextField(SQLStatement, htmlattribute.GetContent());

   SQLStatement << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


///////
   //    Insert a Link into the relative table
   //    Returns 0 if an error occured
///////

int HtmysqlDB::Insert(const Link& link)
{

   String SQLStatement="";
   String LinkType="";
   String LinkResult="";
   
   // Retrieve the String value of the type
   link.RetrieveLinkType(LinkType);   
   
   // Retrieve the String value of the result
   link.RetrieveLinkResult(LinkResult);   
   
   
   if (debug >4)
      cout << "Inserting a new Link into '"
      << MySQLDB << "': " << link << endl;

   SQLStatement << "Insert into " << MySQLDB << ".Link ( "
      << "IDUrlSrc, IDUrlDest, TagPosition, AttrPosition, "
      << "Anchor, LinkType, LinkResult"
      << " ) values ( "
      << link.GetIDUrlSrc() << ", "
      << link.GetIDUrlDest() << ", "
      << link.GetTagPosition() << ", "
      << link.GetAttrPosition() << ", ";

   AppendSQLTextField(SQLStatement, link.GetAnchor());

   SQLStatement << ", "
      << "'" << LinkType << "', "
      << "'" << LinkResult << "'"
      << ")";

   // Executing Insert query

   if (Query (SQLStatement) == -1)
      return 0;   // An error occured

   return 1;
}


// This method just make a string to be escape safe, which means
// that it makes it get ready to be put into the database through
// an INSERT SQL statement. The string character used is ' (single
// quote).

void HtmysqlDB::AppendSQLTextField(String &Dest, const char *source)
{

   Dest.append('\'');
   char prev = 0;
   
   while(*source)
   {
      if (*source == '\\' || *source == '\'')
            Dest.append('\\');

      prev = *source++;
      
      Dest.append(prev);
      
   }      

   Dest.append('\'');

}


// Alter the Link table, by creating the Indexes - not used
// before now (this could save a lot of time when adding new
// link entries while crawling

int HtmysqlDB::CreateLinkTableIndexes()
{

   int result = 0;
   String SQLStatement=0;

   if (debug>0)
      cout << "Create indexes for the Link table" << endl;

   // IDUrlDest index
   SQLStatement << "CREATE INDEX Idx_IDUrlDest ON " 
      << MySQLDB << ".Link " 
      << "(IDUrlDest)" << "\n";

   if (debug>1)
      cout << "   > Idx_IDUrlDest on IDUrlDest" << endl;

   if ((result=Query(SQLStatement)) == -1)
      return result;

   // Anchor index
   SQLStatement.trunc();
   SQLStatement << "CREATE INDEX Idx_Anchor ON " 
      << MySQLDB << ".Link " 
      << "(Anchor(8))" << "\n";

   if (debug>1)
      cout << "   > Idx_Anchor on Anchor" << endl;

   if ((result=Query(SQLStatement)) == -1)
      return result;

   // LinkType index
   SQLStatement.trunc();
   SQLStatement << "CREATE INDEX Idx_LinkType ON "
      << MySQLDB << ".Link "
      << "(LinkType)" << "\n";

   if (debug>1)
      cout << "   > LinkType on LinkType" << endl;

   if ((result=Query(SQLStatement)) == -1)
      return result;

   // LinkResult index
   SQLStatement.trunc();
   SQLStatement << "CREATE INDEX Idx_LinkResult ON " << "\n"
      << "   " << MySQLDB << ".Link " << "\n"
      << "   (LinkResult)" << "\n";

   if (debug>1)
      cout << "   > LinkResult on LinkResult" << endl;

   if ((result=Query(SQLStatement)) == -1)
      return result;

   // Exec the query
   return result;
}


// Create the temporary table for anchors

int HtmysqlDB::CreateAnchorsTable()
{

   // Create the table with all the anchors;

   String SQLFillTemporaryTable=0;
   SQLFillTemporaryTable << "INSERT INTO TmpAnchors "
      << "SELECT HtmlAttribute.IDUrl, HtmlAttribute.TagPosition, "
      << "HtmlAttribute.AttrPosition, HtmlAttribute.Content FROM HtmlAttribute, "
      << "HtmlStatement WHERE HtmlStatement.IDUrl=HtmlAttribute.IDUrl AND "
      << "HtmlAttribute.TagPosition=HtmlStatement.TagPosition AND "
      << "HtmlAttribute.Attribute='name' AND HtmlStatement.Tag='a'";

   if (debug>2)
      cout << "Filling temporary table TmpAnchors" << endl;

   // Exec the query
   return Query(SQLFillTemporaryTable);


}


// Set the result type for anchors

int HtmysqlDB::SetAnchorsResults(Link &LinkTmp)
{

   String strResult;
   LinkTmp.RetrieveLinkResult(strResult);

   String SQLUpdateStatement=0;
   
   SQLUpdateStatement << "Update Link SET LinkResult = '" << strResult << "'"
      << " WHERE IDUrlSrc = " << LinkTmp.GetIDUrlSrc()
      << " AND IDUrlDest = " << LinkTmp.GetIDUrlDest()
      << " AND Anchor = ";

   AppendSQLTextField(SQLUpdateStatement, LinkTmp.GetAnchor());

   return Query(SQLUpdateStatement);

}


#if 0

///////
   //    Look for a schedule entries, given a filter and a
   //     HtmysqlQuery Result object. Returns -1 if an error occurs
   //     else returns the number of records found.
///////

int HtmysqlDB::Search(_Server &filter, HtmysqlQueryResult &result)
{

   String SQLStatement="";

   // SQL statement construction
   
   SQLStatement << "Select "
      << "IDServer"  << ", "     // Server identifier
      << "IPAddress" << ", "     // IP Address
      << "Port" << ", "
      << "HttpServer" << ", "
      << "HttpVersion" << ", "
      << "PersistentConnection" << ", "
      << "Requests"              // Number of requests
      << " from Server ";

   // Create the SQL 'Where' statement given a filter
   CreateFilter(SQLStatement, filter);

   // Execute and store the Query
   
   return Query(SQLStatement, &result);

}



///////
   //    Create a SQL filter string on a _Server Entry
   //    Incomplete ...
///////

void HtmysqlDB::CreateFilter(String &SQLStatement, _Server &filter)
{
   // Search for any filter to be applied to the query
   
   int flag = 0;

   // Applying Server Identifier
      
   if (filter.GetID() != 0)   // specified an ID
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "IDServer = " << filter.GetID() << " ";
   }

   // Applying Server name (host)
      
   if (filter.host() != 0)
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "Server = " << filter.host() << " ";
   }

   // Applying Server port (not done)
      
   // Applying Server Info
      
   if (filter.GetHttpServer() != 0)
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "HttpServer = " << filter.GetHttpServer() << " ";
   }

   // Applying Server Version
      
   if (filter.GetHttpVersion() != 0)
   {
      if (flag)   // already specified a filter
         SQLStatement << "And ";
      else
      {
         SQLStatement << "Where ";
         flag ++; // Add an occurrence to flag
      }

      // Write the sentence
      SQLStatement << "HttpVersion = " << filter.GetHttpVersion() << " ";
   }

}

#endif
