/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iboundeddisksource.h"


#include "ierror.h"
#include "itransform.h"

#include <vtkAppendPolyData.h>
#include <vtkArrowSource.h>
#include <vtkDiskSource.h>
#include <vtkPolyData.h>
#include <vtkPolyDataNormals.h>
#include <vtkTransformPolyDataFilter.h>


//
// iBoundedDiskSource class
//
iBoundedDiskSource* iBoundedDiskSource::New(iViewSubject *vo)
{
	IERROR_ASSERT(vo);
	return new iBoundedDiskSource(vo);
}


iBoundedDiskSource::iBoundedDiskSource(iViewSubject *vo) : iBoundedPolyDataSource(vo,true,true)
{
	mPointNormals = vtkPolyDataNormals::New(); IERROR_ASSERT(mPointNormals);
	mSource = vtkDiskSource::New(); IERROR_ASSERT(mSource);

	mPointNormals->SplittingOff();
	mPointNormals->ConsistencyOff();
	mPointNormals->ComputePointNormalsOn();
	mPointNormals->ComputeCellNormalsOff();
	mPointNormals->FlipNormalsOff();

	mSource->SetInnerRadius(0.0);
	mSource->SetOuterRadius(1.0);

	mAppendPolyData = vtkAppendPolyData::New(); IERROR_ASSERT(mAppendPolyData);
	mArrowSource = vtkArrowSource::New(); IERROR_ASSERT(mArrowSource);
	mTransformPolyDataFilter = vtkTransformPolyDataFilter::New(); IERROR_ASSERT(mTransformPolyDataFilter);
	iTransform *t = iTransform::New(); IERROR_ASSERT(t);
	t->Identity();
	t->RotateY(-90.0);
	mTransformPolyDataFilter->SetTransform(t);
	t->Delete();
	mTransformPolyDataFilter->SetInput(mArrowSource->GetOutput());
	mAppendPolyData->AddInput(mSource->GetOutput());
	mAppendPolyData->AddInput(mTransformPolyDataFilter->GetOutput());

	mWithArrow = false;
	this->SetWithArrow(true);
}


iBoundedDiskSource::~iBoundedDiskSource()
{
	mSource->Delete();
	mPointNormals->Delete();
	mAppendPolyData->Delete();
	mArrowSource->Delete();
	mTransformPolyDataFilter->Delete();
}


void iBoundedDiskSource::SetWithArrow(bool s)
{
	if(s == mWithArrow) return;

	mWithArrow = s;
	if(s)
	{
		mFilter->SetInput(mAppendPolyData->GetOutput());
	}
	else
	{
		mFilter->SetInput(mSource->GetOutput());
	}
}


void iBoundedDiskSource::AddObserverToSource(unsigned long e, vtkCommand *c, float p)
{
	mSource->AddObserver(e,c,p);
	mPointNormals->AddObserver(e,c,p);
}


void iBoundedDiskSource::UpdateSourceResolution()
{
	mSource->SetRadialResolution(1+mResolution/2);
	mSource->SetCircumferentialResolution(3*mResolution);
}


float iBoundedDiskSource::GetSourceMemorySize() const
{
	float s = 0.0;
	s += mPointNormals->GetOutput()->GetActualMemorySize();
	s += mSource->GetOutput()->GetActualMemorySize();
	return s;
}


void iBoundedDiskSource::UpdateBoundaryConditions()
{
}

