/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icolorbars.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatastretch.h"
#include "ierror.h"
#include "igenericprop.h"
#include "imath.h"
#include "ioverlayhelper.h"
#include "ipaletteset.h"
#include "itextactor.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkLookupTable.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkRenderer.h>
#include <vtkScalarBarActor.h>

using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"
#include "igenericproptemplate.h"


IOBJECT_DEFINE_TYPE(iColorBars,ColorBars,cb,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iColorBars,Automatic,a,Bool,1);
IOBJECT_DEFINE_KEY(iColorBars,BarLeft,bl,Int,3);
IOBJECT_DEFINE_KEY(iColorBars,BarRight,br,Int,3);
IOBJECT_DEFINE_KEY(iColorBars,Color,c,Color,1);
IOBJECT_DEFINE_KEY(iColorBars,Size,s,Float,1);
IOBJECT_DEFINE_KEY(iColorBars,SideOffset,so,Float,1);


//
//  helper classes
//
class iColorBarLabel
{
	
	friend class iColorBarActor;

private:

	iColorBarLabel(bool left, iRenderTool *parent)
	{
		Base = iTextActor::New(parent); IERROR_ASSERT(Base);
		Power = iTextActor::New(parent); IERROR_ASSERT(Power);
		Base->SetJustification(left?1:-1);
		Power->SetJustification(-1);
	}

	~iColorBarLabel()
	{
		Base->Delete();
		Power->Delete();
	}

	iTextActor *Base, *Power;
};


class iColorBarActor : public iGenericProp<vtkActor2D>
{
	
	IPOINTER_AS_USER(OverlayHelper);

public:
	
	vtkTypeMacro(iColorBarActor,vtkActor2D);
	static iColorBarActor* New(bool left = false, iColorBars *parent = 0);

	void SetSideOffset(float v);
	inline float GetSideOffset() const { return mOffset; }

	void SetHeight(float v);
	inline float GetHeight() const { return mHeight*0.86; } // there is a 0.86 factor in vtkScalarBarActor

protected:

	virtual ~iColorBarActor();

	virtual void UpdateGeometry(vtkViewport *vp);

private:
	
	iColorBarActor(bool left, iColorBars *parent);

	void UpdatePlacement();

	const iColorBarItem *mItem;

	bool mStarted;

	float mHeight, mOffset, mWidth4Height;
	float mXOff, mYOff;
	float mLabelOff, mTitleOff;
	int mJmax;

	bool mIsLeft;
	iColorBars *mParent;
	vtkScalarBarActor *mBarActor;
	iTextActor *mTitleActor;
	iArray<iColorBarLabel*> mLabels;
};


iColorBarActor* iColorBarActor::New(bool left, iColorBars *parent)
{
	IERROR_ASSERT(parent);
	return new iColorBarActor(left,parent);
}


iColorBarActor::iColorBarActor(bool left, iColorBars *parent) : iGenericProp<vtkActor2D>(false), mOverlayHelper(parent->GetViewModule()->GetRenderTool())
{
	int i;

	mIsLeft = left;
	mParent = parent;
	mItem = 0;

	mStarted = false;

	mJmax = 0;
	mWidth4Height = 0.03/0.7;
	mHeight = 0.7;
	mOffset = 0.08;

	mBarActor = vtkScalarBarActor::New(); IERROR_ASSERT(mBarActor);
	this->AppendComponent(mBarActor);
	mBarActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	mBarActor->SetOrientationToVertical();
	mBarActor->SetMaximumNumberOfColors(256);
	mBarActor->SetWidth(mWidth4Height*mHeight);
	mBarActor->SetHeight(mHeight);
	mBarActor->SetNumberOfLabels(0);
	mBarActor->SetTitle("");
	mBarActor->PickableOff();

	mTitleActor = iTextActor::New(this->GetOverlayHelper()->GetRenderTool()); IERROR_ASSERT(mTitleActor);
	this->AppendComponent(mTitleActor);
	mTitleActor->SetAngle(90.0);
	mTitleActor->SetJustification(left?-1:1,0);
	mTitleActor->SetBold(true);


	iColorBarLabel *l;
	for(i=0; i<10; i++)
	{
		l = new iColorBarLabel(left,this->GetOverlayHelper()->GetRenderTool()); IERROR_ASSERT(l);
		this->AppendComponent(l->Base);
		this->AppendComponent(l->Power);
		mLabels.Add(l);
	}

	this->UpdatePlacement();
}


iColorBarActor::~iColorBarActor()
{
	int i;

	mBarActor->Delete();
	mTitleActor->Delete();
	for(i=0; i<mLabels.Size(); i++) delete mLabels[i];
}


void iColorBarActor::UpdatePlacement()
{
	if(mIsLeft)
	{
		mXOff = mOffset; 
		mLabelOff = mXOff - 0.01;
		mTitleOff = mXOff + mWidth4Height*mHeight;
	}
	else
	{
		mXOff = 1.0 - mWidth4Height*mHeight - mOffset;
		mLabelOff = mXOff + mWidth4Height*mHeight;
		mTitleOff = mXOff - 0.3*mWidth4Height*mHeight;
	}
	mYOff = 0.01 + 0.5*(1-mHeight*0.86);  // there is a 0.86 factor in vtkScalarBarActor
	mTitleActor->SetPosition(mTitleOff,0.5);
	this->Modified();
}


void iColorBarActor::SetSideOffset(float v)
{
	if(v>=0.01 && v<=0.3)
	{
		mOffset = v;
		this->UpdatePlacement();
	}
}


void iColorBarActor::SetHeight(float v)
{
	if(v>=0.1 && v<=1.0)
	{
		if(v > 0.95) v = 0.95;  //  make sure labels are visible;
		mHeight = v/0.86;       // there is a 0.86 factor in vtkScalarBarActor
		this->UpdatePlacement();
	}
}


void iColorBarActor::UpdateGeometry(vtkViewport* viewport)
{
	mItem = mParent->GetItem(mIsLeft);
	if(mItem==0 || mItem->Count==0 || mItem->Palette==0)
	{
		this->Disable();
		return;
	}

	iDataLimits *lim = mParent->GetViewModule()->GetReader()->GetLimits(iDataType::FindTypeById(mItem->DataTypeId));
	if(lim==0 || mItem->Var>=lim->GetNumVars())
	{
		this->Disable();
		return;
	}

	int j;
	float w1 = 0.0, h1 = 0.0, w2 = 0.0, h2 = 0.0;
	if(!mStarted) mStarted = true;

	int mag = this->GetOverlayHelper()->GetRenderingMagnification();
	if(mag == 1)
	{
		mBarActor->SetPosition(mXOff,mYOff);
		mBarActor->SetWidth(mWidth4Height*mHeight);
		mBarActor->SetHeight(mHeight);

		int nstep, pow;
		float dv, voff, vstep, vbound, v;
		float s1[2], s2[2];
		char s[100], *fmt;

		float vl = iDataStretch::ApplyStretch(lim->GetLowerLimit(mItem->Var),lim->GetStretch(mItem->Var),false);
		float vu = iDataStretch::ApplyStretch(lim->GetUpperLimit(mItem->Var),lim->GetStretch(mItem->Var),true);
		switch(lim->GetStretch(mItem->Var))
		{
		case iDataStretch::Log:
			{
				//
				//  Log table
				//
				int ivl = 1 + (int)floor(vl-0.01);
				int ivu = 0 + (int)floor(vu+0.01);
				dv = vu - vl;

				if(dv<=0.0 || ivl>vu)
				{
					this->Disable();
					return;
				}

				voff = ivl - vl;
				int ndex = ivu - ivl;
				if(ndex > 1) nstep = 1 + ndex/mLabels.Size(); else nstep = 1;

				mJmax = 1 + ndex/nstep;
				vstep = nstep;

				for(j=0; j<mJmax; j++)
				{
					pow = ivl + nstep*j;
					sprintf(s,"%d",pow);

					if(pow == 0)
					{
						mLabels[j]->Base->SetText(" 1"); 
						mLabels[j]->Power->SetText(""); 
					} 
					else 
					{
						mLabels[j]->Base->SetText("10"); 
						mLabels[j]->Power->SetText(s);
					}

					mLabels[j]->Base->GetSize(viewport,s1);
					mLabels[j]->Power->GetSize(viewport,s2);
					if(w1 < s1[0]) w1 = s1[0];
					if(h1 < s1[1]) h1 = s1[1];
					if(w2 < s2[0]) w2 = s2[0];
					if(h2 < s2[1]) h2 = s2[1];	
				}
				break;
			}
		default:
			{
				//
				//  Lin table
				//
				mJmax = 5;
				voff = 0.0;
				if(fabs(vl) < 1.0e-3*fabs(vu)) vl = 0.0;
				if(fabs(vu) < 1.0e-3*fabs(vl)) vu = 0.0;
				dv = vu - vl;
				if(fabs(dv) < 1.0e-30*fabs(vu)) // dv is zero
				{
					if(vu > 0.0)
					{
						vu = vu*1.001;
						vl = vl*0.999;
					}
					else if(vu < 0.0)
					{
						vu = vu*0.999;
						vl = vl*1.001;
					}
					else
					{
						vu = 1.0;
						vl = -1.0;
					}
					dv = vu - vl;
				}
				vstep = dv/(mJmax-1);
				vbound = (fabs(vl) > fabs(vu)) ? fabs(vl) : fabs(vu);
				if(fabs(vstep) > 0.03*vbound)
				{
					fmt = "%.2g";
				}
				else if(fabs(vstep) > 0.003*vbound)
				{
					fmt = "%.3g";
				}
				else
				{
					fmt = "%g";
				}

				for(j=0; j<mJmax; j++)
				{
					v = vl + vstep*j;
					sprintf(s,fmt,v);

					mLabels[j]->Base->SetText(s); 
					mLabels[j]->Power->SetText(""); 

					mLabels[j]->Base->GetSize(viewport,s1);
					mLabels[j]->Power->GetSize(viewport,s2);
					if(w1 < s1[0]) w1 = s1[0];
					if(h1 < s1[1]) h1 = s1[1];
					if(w2 < s2[0]) w2 = s2[0];
					if(h2 < s2[1]) h2 = s2[1];
				}
			}
		}

		w1 *= 0.8;
		w2 *= 0.8;
		h1 *= 0.8;

		float xpos, ypos;
		xpos = mLabelOff;
		for(j=0; j<mJmax; j++)
		{
			if(dv > 0.0)
			{
				ypos = mYOff + 0.86*mHeight*(voff+vstep*j)/dv;
			}
			else
			{
				ypos = mYOff + 0.86*mHeight*0.5;
			}

			if(!mIsLeft) 
			{
				mLabels[j]->Base->SetPosition(xpos,ypos-0.5*h1);
				mLabels[j]->Power->SetPosition(w1+xpos,ypos+0.5*h1);
			} 
			else 
			{
				mLabels[j]->Base->SetPosition(xpos-w2,ypos-0.5*h1);
				mLabels[j]->Power->SetPosition(xpos-w2,ypos+0.5*h1);
			}
		}

		for(j=mJmax; j<mLabels.Size(); j++)
		{
			mLabels[j]->Base->SetText(""); 
			mLabels[j]->Power->SetText(""); 
		}

		mTitleActor->SetText(lim->GetName(mItem->Var).ToCharPointer());
		mBarActor->SetLookupTable(iPaletteSet::Default()->GetLookupTable(mItem->Palette));
	}
	else
	{
		mBarActor->SetWidth(mag*mWidth4Height*mHeight);
		mBarActor->SetHeight(mag*mHeight);

		//
		//  Shift positions if under magnification - 
		//
		int winij[2];
		this->GetOverlayHelper()->ComputePositionShiftsUnderMagnification(winij);

		mBarActor->SetPosition(mag*mXOff-winij[0],mag*mYOff-winij[1]);
	}
}


const struct iColorBarItem iColorBars::NullItem = { 0, 0, 0, iMath::_IntMax, 0 };

//
//  Main class
//
iColorBars* iColorBars::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iColorBars(vm);
}


iColorBars::iColorBars(iViewModule *vm) : iObject("ColorBars"), mViewModule(vm)
{
	mAutomatic = true;

	//
	//  Create manual bars
	//
	mManualBars.Add(NullItem);
	mManualBars.Add(NullItem);
	mManualBars[0].Priority = -2;
	mManualBars[1].Priority = -1;
	mManualBars[0].DataTypeId = mManualBars[1].DataTypeId = 1;
	mManualBars[0].Count = mManualBars[1].Count = 1;

	mActors[0] = iColorBarActor::New(false,this); IERROR_ASSERT(mActors[0]);
	mActors[1] = iColorBarActor::New(true,this); IERROR_ASSERT(mActors[1]);

	mColor = iColor::Invalid();
}


iColorBars::~iColorBars()
{
	int i;
	for(i=0; i<2; i++) mActors[i]->Delete();
}


void iColorBars::SetSideOffset(float v)
{
	int i;
	for(i=0; i<2; i++) mActors[i]->SetSideOffset(v);
	this->ClearCache();
}


float iColorBars::GetSideOffset() const
{
	return mActors[1]->GetSideOffset();
}


void iColorBars::SetSize(float v)
{
	int i;
	for(i=0; i<2; i++) mActors[i]->SetHeight(v);
	this->ClearCache();
}


float iColorBars::GetSize() const
{
	return mActors[1]->GetHeight();
}


void iColorBars::SetColor(iColor c)
{
	int i;

	mColor = c;

	for(i=0; i<2; i++)
	{
		mActors[i]->GetOverlayHelper()->SetFixedColor(c);
		mActors[i]->GetOverlayHelper()->SetAutoColor(!c.IsValid());
	}

	this->ClearCache();
}

	
void iColorBars::ShowBar(int priority, int v, const iDataType &dt, int p, bool show)
{
	int i;
	
	if((!show && this->Queue().Size()<1) || v<0) return;

	iDataLimits *lim = this->GetViewModule()->GetReader()->GetLimits(dt);
	if(lim==0 || (show && v>=lim->GetNumVars())) return;

	iColorBarItem tmp;
	tmp.Var = v;
	tmp.Palette = p;
	tmp.DataTypeId = dt.GetId();
	tmp.Priority = priority;
	i = this->Queue().Find(tmp);

	if(i >= 0) 
	{
		if(show) this->Queue()[i].Count++; else 
		{
			this->Queue()[i].Count--;
			if(this->Queue()[i].Count == 0) // delete the entry
			{
				this->Queue().iArray<iColorBarItem>::Remove(i);
			}
		}
	}
	else 
	{
		if(show)
		{
			//
			//  Insert a new one keeping the list ordered by priority
			//
			tmp.Count = 1;
			this->Queue().Add(tmp);
		}
		else
		{
			// cannot delete non-existing entry
		}
	}
	this->ClearCache();
	for(i=0; i<2; i++) mActors[i]->Modified();
}


void iColorBars::SetAutomatic(bool automatic)
{
	int i;
	mAutomatic = automatic;
	this->ClearCache();
	for(i=0; i<2; i++) mActors[i]->Modified();
}


void iColorBars::SetBar(int bar, int v, int id, int p)
{
	if(bar>=0 && bar<2)
	{
		mManualBars[bar].Var = v;
		mManualBars[bar].Palette = p;
		mManualBars[bar].DataTypeId = id;
		this->ClearCache();
	}
}


void iColorBars::GetBar(int bar, int &v, int &id, int &p) const
{
	if(bar<0 || bar>1)
	{
		v = id = p = 0;
	}
	else
	{
		v = mManualBars[bar].Var;
		p = mManualBars[bar].Palette;
		id = mManualBars[bar].DataTypeId;
	}
}

//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iColorBars::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyColor(),mColor);
	this->PackValue(s,KeySideOffset(),this->GetSideOffset());
	this->PackValue(s,KeySize(),this->GetSize());
	this->PackValue(s,KeyAutomatic(),this->GetAutomatic());

	int iv[3];
	this->GetBar(0,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	this->PackValue(s,KeyBarRight(),iv,3);
	this->GetBar(1,iv[0],iv[1],iv[2]);
	iv[0]++;
	iv[2]++;
	this->PackValue(s,KeyBarLeft(),iv,3);
}


void iColorBars::UnPackStateBody(const iString &s)
{
	int iv[3]; bool a; float f; iColor c;

	if(this->UnPackValue(s,KeyColor(),c)) this->SetColor(c);
	if(this->UnPackValue(s,KeySideOffset(),f)) this->SetSideOffset(f);
	if(this->UnPackValue(s,KeySize(),f)) this->SetSize(f);

	a = this->GetAutomatic();
	this->UnPackValue(s,KeyAutomatic(),a);
	this->SetAutomatic(false);

	this->GetBar(0,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	if(this->UnPackValue(s,KeyBarRight(),iv,3)) this->SetBar(0,iv[0]-1,iv[1],iv[2]-1);
	this->GetBar(1,iv[0],iv[1],iv[2]);
	iv[0]++;  //  vars start with 1
	iv[2]++;
	if(this->UnPackValue(s,KeyBarLeft(),iv,3)) this->SetBar(1,iv[0]-1,iv[1],iv[2]-1);

	this->SetAutomatic(a);
}


const iColorBarItem* iColorBars::GetItem(bool left)
{
	int i = left ? 1 : 0;
	if(this->Queue().Size() > i) return &(this->Queue()[i]); else return 0; 
}


vtkActor2D* iColorBars::GetActor(bool left) const
{
	return mActors[left?1:0];
}


bool iColorBarItem::operator<(const iColorBarItem &item) const
{
	return Priority < item.Priority;
}


bool iColorBarItem::operator==(const iColorBarItem &item) const
{
	return (Var==item.Var && DataTypeId==item.DataTypeId && Palette==item.Palette); 
}

