/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IRANGECOLLECTION_H
#define IRANGECOLLECTION_H


#include "idatastretch.h"


#include "iarray.h"


class iRangeCollection : public iDataStretchUser
{
	
public:
	
	class Range
	{
	public:
		Range(){ Min = 0.0; Max = 0.0; }
		Range(float min, float max){ Min = min; Max = max; }
		float Min, Max;
	};

	iRangeCollection(float min = 0.0, float max = 1.0);
	~iRangeCollection();
	
	inline int GetN() const { return mArr.Size(); }
	float GetMin(int i) const;
	float GetMax(int i) const;
	float GetGlobalMin() const;
	float GetGlobalMax() const;
	void GetRange(int i, float &min, float &max) const;

	//
	//  Functions that return stretched values
	//
	float GetStretchedMin(int i) const;
	float GetStretchedMax(int i) const;
	float GetStretchedGlobalMin() const;
	float GetStretchedGlobalMax() const;

	void SetGlobalRange(float min, float max);
	void SetGlobalMin(float min);
	void SetGlobalMax(float max);

	void SetRange(int i, float min, float max);
	void SetMin(int i, float min);
	void SetMax(int i, float max);

	void SetTiled(bool s);
	inline bool GetTiled() const { return mTiled; }

	void AddRange();
	void AddRange(float min, float max);
	bool RemoveRange(int n);

	void Copy(const iRangeCollection *f);
	void Reset();
	
private:
	
	iArray<Range> mArr;
	float mGlobalMin, mGlobalMax;
	bool mTiled;
};


inline float iRangeCollection::GetMin(int i) const
{
	if(i>=0 && i<mArr.Size()) return mArr[i].Min; else return mGlobalMin;
}


inline float iRangeCollection::GetMax(int i) const
{
	if(i>=0 && i<mArr.Size()) return mArr[i].Max; else return mGlobalMax;
}


inline float iRangeCollection::GetGlobalMin() const
{
	return mGlobalMin;
}


inline float iRangeCollection::GetGlobalMax() const
{
	return mGlobalMax;
}


inline float iRangeCollection::GetStretchedMin(int i) const
{
	if(i>=0 && i<mArr.Size()) return this->ApplyStretch(mArr[i].Min,false); else return this->ApplyStretch(mGlobalMin,false);
}


inline float iRangeCollection::GetStretchedMax(int i) const
{
	if(i>=0 && i<mArr.Size()) return this->ApplyStretch(mArr[i].Max,true); else return this->ApplyStretch(mGlobalMax,true);
}


inline float iRangeCollection::GetStretchedGlobalMin() const
{
	return this->ApplyStretch(mGlobalMin,false);
}


inline float iRangeCollection::GetStretchedGlobalMax() const
{
	return this->ApplyStretch(mGlobalMax,true);
}


inline void iRangeCollection::GetRange(int i, float &min, float &max) const
{
	if(i>=0 && i<mArr.Size())
	{
		min = mArr[i].Min;
		max = mArr[i].Max;
	}
}


inline void iRangeCollection::SetGlobalMin(float min)
{
	this->SetGlobalRange(min,mGlobalMax);
}


inline void iRangeCollection::SetGlobalMax(float max)
{
	this->SetGlobalRange(mGlobalMin,max);
}


inline void iRangeCollection::SetMin(int i, float min)
{
	this->SetRange(i,min,this->GetMax(i));
}


inline void iRangeCollection::SetMax(int i, float max)
{
	this->SetRange(i,this->GetMin(i),max);
}

#endif // IRANGECOLLECTION_H

