/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivectorfieldstreamlinepipeline.h"


#include "idatalimits.h"
#include "ierror.h"
#include "ilinetobandfilter.h"
#include "ilinetotubefilter.h"
#include "ireplicatedpolydata.h"
#include "istreamlinefilter.h"
#include "ivectorfieldviewsubject.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"
#include "iviewsubjectpipelinetemplate.h"


using namespace iParameter;


//
// iVectorFieldStreamLinePipeline class
//
iVectorFieldStreamLinePipeline::iVectorFieldStreamLinePipeline(iVectorFieldViewSubject *s) : iViewSubjectPipeline(s,2)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mLine = this->CreateFilter<iStreamLineFilter>();
	mTubeFilter = this->CreateFilter<iLineToTubeFilter>();
	mBandFilter = this->CreateFilter<iLineToBandFilter>();
	mDataReplicated = this->CreateFilter<iReplicatedPolyData>();
	
	mTubeFilter->SetNumberOfSides(6);
	mTubeFilter->SetVaryRadiusToVaryRadiusByScalar();

	mBandFilter->SetDistanceFactor(15.0);
	mBandFilter->SetRuledModeToPointWalk();
	mBandFilter->SetOffset(0);
	mBandFilter->SetOnRatio(2);
	mBandFilter->PassLinesOn();

	this->UpdateContents(_All);
}


iVectorFieldStreamLinePipeline::~iVectorFieldStreamLinePipeline()
{
}


void iVectorFieldStreamLinePipeline::ProduceOutput()
{
	vtkDataSet *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();
	vtkDataSet *data = vtkDataSet::SafeDownCast(this->GetInputs()[1]);
	if(data == 0)
	{
		output->Initialize();
		return;
	}

	if(mLine->GetSource() != input)
	{
		mLine->SetSource(input);
	}
	if(mLine->GetInput() != data)
	{
		mLine->SetInput(data);
	}

	//
	//  Update the pipeline
	//
	mDataReplicated->Update();
	output->ShallowCopy(mDataReplicated->GetOutput());
}


float iVectorFieldStreamLinePipeline::GetContentsMemorySize() const
{
	float s = 0.0f;
	s += mLine->GetMemorySize();
	s += mTubeFilter->GetMemorySize();
	s += mDataReplicated->GetMemorySize();
	s += mBandFilter->GetMemorySize();
	return s;
}


void iVectorFieldStreamLinePipeline::UpdateContents(int n, int)
{
	if(n==_All || n==_Direction) this->UpdateDirection();
	if(n==_All || n==_Length) this->UpdateLength();
	if(n==_All || n==_Pipeline) this->UpdatePipeline();
	if(n==_All || n==_Quality) this->UpdateQuality();
	if(n==_All || n==_Replicas) this->UpdateReplicas();
	if(n==_All || n==_TubeRangeFactor) this->UpdateTubeRangeFactor();
	if(n==_All || n==_TubeSize) this->UpdateTubeSize();
	if(n==_All || n==_TubeVariationFactor) this->UpdateTubeVariationFactor();
}


void iVectorFieldStreamLinePipeline::UpdatePipeline()
{
	switch(mParent->GetMethod())
	{
	case VectorField::Method::Glyph:
		{
			//
			//  mode=0 is handled by iVectorFieldPipeline1, ignore
			//
			break;
		}
	case VectorField::Method::StreamLine:
		{
			mLine->SetSplitLines(false);
			mDataReplicated->SetInput(mLine->GetOutput()); 
			break;
		}
	case VectorField::Method::StreamTube:
		{
			mLine->SetSplitLines(false);
			mTubeFilter->SetInput(mLine->GetOutput());
			mDataReplicated->SetInput(mTubeFilter->GetOutput());
			break;
		}
	case VectorField::Method::StreamBand:
		{
			mLine->SetSplitLines(true);
			mBandFilter->SetInput(mLine->GetOutput());
			mDataReplicated->SetInput(mBandFilter->GetOutput());
			break;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
		}
	}
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateDirection()
{ 
	mLine->SetDirection(mParent->GetLineDir());
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateLength()
{ 
	mLine->SetLength(2.0*mParent->GetLineLength());
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateQuality()
{ 
	mLine->SetQuality(mParent->GetLineQuality());
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateTubeSize()
{ 
	mTubeFilter->SetRadius(0.005*mParent->GetTubeSize());
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateTubeRangeFactor()
{ 
	mTubeFilter->SetRadiusFactor(mParent->GetTubeRangeFactor());
	this->Modified();
}


void iVectorFieldStreamLinePipeline::UpdateTubeVariationFactor()
{
	//
	//  Do not update unless necessary
	//
	if(mParent->CanBeShown())
	{
		mLine->SetMinimumSpeed(mParent->GetTubeVariationFactor()*this->GetLimits()->GetUpperLimit(mParent->GetDataOffset()));
		this->Modified();
	}
}


void iVectorFieldStreamLinePipeline::UpdateReplicas()
{
	mDataReplicated->Modified();
	this->Modified();
}


void iVectorFieldStreamLinePipeline::SyncWithData(const iDataSyncRequest &)
{
	this->UpdateTubeVariationFactor();
}

