#ifndef TAGCOLL_COMMANDLINE_H
#define TAGCOLL_COMMANDLINE_H

#include <tagcoll/Exception.h>
#include <string>
#include <vector>
#include <list>
#include <map>
#include <ostream>

namespace Tagcoll {
namespace commandline {

typedef std::list<const char*> arglist;
typedef arglist::iterator iter;

class BadOption : public ConsistencyCheckException
{
public:
	BadOption(const std::string& context) throw ()
		: ConsistencyCheckException(context) {}
	virtual ~BadOption() throw () {}

	virtual const char* type() const throw () { return "BadOption"; }
};

/// Interface for everything that would parse an arglist
class Parser
{
	std::string m_name;

public:
	Parser(const std::string& name) : m_name(name) {}
	virtual ~Parser() {}
	
	const std::string& name() const { return m_name; }

	/**
	 * Parse the list of arguments, starting at the beginning and removing the
	 * arguments it successfully parses.
	 *
	 * @returns
	 *   An iterator to the first unparsed argument (can be list.end())
	 */
	virtual iter parseList(arglist& list) { return parse(list, list.begin()); }

	/**
	 * Parse the list of arguments, starting at 'begin' and removing the
	 * arguments it successfully parses.
	 *
	 * The 'begin' iterator can be invalidated by this function.
	 *
	 * @returns
	 *   An iterator to the first unparsed argument (can be list.end())
	 */
	virtual iter parse(arglist& list, iter begin) = 0;
};

/// Interface for a parser for one commandline option
class Option
{
	std::string m_name;
	mutable std::string m_fullUsage;

public:
	Option(const std::string& name) : m_name(name) {}
	Option(const std::string& name, char shortName, const std::string& longName) : m_name(name)
	{
		if (shortName != 0)
			shortNames.push_back(shortName);
		if (!longName.empty())
			longNames.push_back(longName);
	}
	virtual ~Option() {}

	const std::string& name() const { return m_name; }

	void addAlias(char c) { shortNames.push_back(c); }
	void addAlias(const std::string& str) { longNames.push_back(str); }

	virtual bool boolValue() const = 0;
	virtual std::string stringValue() const = 0;
	virtual int intValue() const;

	/**
	 * Signal that the option has been found, with the given argument (or 0 if
	 * no argument).
	 *
	 * @returns
	 *   true if it used the argument, else false
	 */
	virtual bool parse(const char* str = 0) = 0;

	/// Return a full usage message including all the aliases for this option
	const std::string& fullUsage() const;
	std::string fullUsageForMan() const;

	std::vector<char> shortNames;
	std::vector<std::string> longNames;

	std::string usage;
	std::string description;
};

/// Boolean option
class BoolOption : public Option
{
	bool m_value;
public:
	BoolOption(const std::string& name)
		: Option(name), m_value(false) {}
	BoolOption(const std::string& name, char shortName, const std::string& longName)
		: Option(name, shortName, longName), m_value(false) {}

	bool boolValue() const { return m_value; }
	std::string stringValue() const { return m_value ? "true" : "false"; }

	bool parse(const char* str) { m_value = true; return false; }
};

// Option needing a compulsory string value
class StringOption : public Option
{
	std::string m_value;
public:
	StringOption(const std::string& name)
		: Option(name)
	{
		usage = "<val>";
	}
	StringOption(const std::string& name, char shortName, const std::string& longName)
		: Option(name, shortName, longName)
	{
		usage = "<val>";
	}

	bool boolValue() const { return !m_value.empty(); }
	std::string stringValue() const { return m_value; }

	bool parse(const char* str);
};

// Option needing a compulsory int value
class IntOption : public Option
{
	bool m_has_value;
	int m_value;

public:
	IntOption(const std::string& name)
		: Option(name), m_has_value(false), m_value(0)
	{
		usage = "<num>";
	}
	IntOption(const std::string& name, char shortName, const std::string& longName)
		: Option(name, shortName, longName), m_has_value(false), m_value(0)
	{
		usage = "<num>";
	}

	bool boolValue() const { return m_has_value; }
	int intValue() const { return m_value; }
	std::string stringValue() const;

	bool parse(const char* str);
};

class ExistingFileOption : public Option
{
	std::string m_value;
public:
	ExistingFileOption(const std::string& name)
		: Option(name)
	{
		usage = "<file>";
	}
	ExistingFileOption(const std::string& name, char shortName, const std::string& longName)
		: Option(name, shortName, longName)
	{
		usage = "<file>";
	}

	bool boolValue() const { return !m_value.empty(); }
	std::string stringValue() const { return m_value; }

	bool parse(const char* str);
};

class OptionGroup
{

public:
	void add(Option* o) { options.push_back(o); }

	std::vector<Option*> options;

	std::string description;
};

/// Parser of many short or long switches all starting with '-'
class OptionParser : public Parser
{
	std::map<char, Option*> m_short;
	std::map<std::string, Option*> m_long;
	std::vector<OptionGroup*> m_groups;
	std::vector<Option*> m_options;

	/// Parse a consecutive sequence of switches
	iter parseConsecutiveSwitches(arglist& list, iter begin);

	void addWithoutAna(Option* o);

public:
	OptionParser(const std::string& name)
		: Parser(name), primaryAlias(name) {}

	void add(Option* o);
	void add(OptionGroup* group);

	const std::vector<OptionGroup*>& groups() const { return m_groups; }
	const std::vector<Option*>& options() const { return m_options; }

	/**
	 * Parse all the switches in list, leaving only the non-switch arguments or
	 * the arguments following "--"
	 */
	virtual iter parse(arglist& list, iter begin);

	std::string primaryAlias;
	std::vector<std::string> aliases;
	std::string usage;
	std::string description;
	std::string longDescription;
	std::string examples;
};

class CommandParser : public Parser
{
	OptionParser* m_last_command;
	std::map<std::string, OptionParser*> m_aliases;

	void add(const std::string& alias, OptionParser* o);

public:
	CommandParser(const std::string& name)
		: Parser(name), m_last_command(0) {}

	OptionParser* lastCommand() const { return m_last_command; }
	OptionParser* command(const std::string& name) const;

	void add(OptionParser& o);

	/**
	 * Look for a command as the first non-switch parameter found, then invoke
	 * the corresponding switch parser.
	 *
	 * After this function, only non-switch arguments will be left in list
	 *
	 * If no commands have been found, returns begin.
	 */
	virtual iter parse(arglist& list, iter begin);

	std::map<std::string, OptionParser*> getCommandInfo() const;

	std::string usage;
	std::string description;
	std::string longDescription;
};

/**
 * Main parser for commandline arguments.
 *
 * It is implemented as a template, to allow to base it on top of any available
 * commandline parser.
 */
template<class Base>
class MainParser : public Base
{
	arglist args;

public:
	MainParser(const std::string& name) : Base(name) {}

	arglist parse(int argc, const char* argv[])
	{
		for (int i = 1; i < argc; i++)
			args.push_back(argv[i]);
		this->parseList(args);
		return args;
	}

	bool hasNext() const { return !args.empty(); }

	std::string next()
	{
		if (args.empty())
			return std::string();
		std::string res(*args.begin());
		args.erase(args.begin());
		return res;
	}
};

class DocMaker
{
protected:
	std::string m_app;
	std::string m_ver;

public:
	DocMaker(const std::string& app, const std::string& ver)
		: m_app(app), m_ver(ver) {}
};

class Help : public DocMaker
{
public:
	Help(const std::string& app, const std::string& ver)
		: DocMaker(app, ver) {}
	
	void outputVersion(std::ostream& out);
	void outputHelp(std::ostream& out, const CommandParser& cp);
	void outputHelp(std::ostream& out, const OptionParser& cp);
};

class Manpage : public DocMaker
{
public:
	enum where { BEFORE, BEGINNING, END };

private:
	struct Hook
	{
		std::string section;
		where placement;
		std::string text;

		Hook(const std::string& section, where placement, const std::string& text)
			: section(section), placement(placement), text(text) {}
	};

	std::vector<Hook> hooks;
	std::string lastSection;
	
	void outputParagraph(std::ostream& out, const std::string& str);
	void outputOption(std::ostream& out, const Option* o);
	void runHooks(std::ostream& out, const std::string& section, where where);
	void startSection(std::ostream& out, const std::string& name);
	void endSection(std::ostream& out);


public:
	Manpage(const std::string& app, const std::string& ver)
		: DocMaker(app, ver) {}

	void addHook(const std::string& section, where placement, const std::string& text)
	{
		hooks.push_back(Hook(section, placement, text));
	}
	void readHooks(const std::string& file);

	void output(std::ostream& out, const CommandParser& cp, int section);
	void output(std::ostream& out, const OptionParser& cp, int section);
};

}
}

// vim:set ts=4 sw=4:
#endif
