/*
 * tagged collection - Experimental programs to test and study tagged collections
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "TaggrepOptions.h"

#include <stdio.h>

#include <stdlib.h>	// getenv

#include <errno.h>

#include <tagcoll/stringf.h>
#include <tagcoll/Exception.h>
#include <tagcoll/Consumer.h>
#include <tagcoll/StdioParserInput.h>
#include <tagcoll/TextFormat.h>
#include <tagcoll/Expression.h>

#include <algorithm>
#include <iostream>

using namespace std;
using namespace Tagcoll;

void readCollection(const string& file, Consumer<string, string>& builder)
	throw (FileException, ParserException)
{
	TrivialConverter<string, string> conv;

	if (file == "-")
	{
		StdioParserInput input(stdin, "<stdin>");
		TextFormat<string, string>::parse(conv, conv, input, builder);
	}
	else
	{
		StdioParserInput input(file);
		TextFormat<string, string>::parse(conv, conv, input, builder);
	}
}

int main(int argc, const char* argv[])
{
	commandline::TaggrepOptions opts;

	try {
		opts.parse(argc, argv);

		if (opts.helpGroup.help->boolValue())
		{
			// Provide help as requested
			commandline::Help help("taggrep", VERSION);
			help.outputHelp(cout, opts);
		}
		else if (opts.helpGroup.version->boolValue())
		{
			// Print the program version
			commandline::Help help("taggrep", VERSION);
			help.outputVersion(cout);
		}
		else
		{
			string expression = opts.next();

			FilterChain<string, string> filters;

			FilterItemsByExpression<string, string> filter(expression);
			filters.appendFilter(filter);

			if (opts.invert->boolValue())
				filter.setMatchType(FilterItemsByExpression<string, string>::INVERTED);

			TrivialConverter<string, string> conv;
			Sink<string, string> sink;
			TextFormat<string, string> output(conv, conv, stdout);
			if (opts.quiet->boolValue())
				filters.setConsumer(sink);
			else
				filters.setConsumer(output);

			do
			{
				string file = opts.next();
				readCollection(file, filters);
			}
			while (opts.hasNext());

			return filter.countMatched() > 0 ? 0 : 1;
		}
		return 0;
	} catch (commandline::BadOption& e) {
		cerr << e.desc() << endl;
		commandline::Help help(APPNAME, VERSION);
		help.outputHelp(cerr, opts);
		exit(1);
	} catch (Exception& e)
	{
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		return 1;
	}
}

// vim:set ts=4 sw=4:
